%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This work by EPFL STI IBI LBNI is licensed under 
% a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
% Based on a work at http://lbni.epfl.ch/.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%% Set the GUI Layout
function [] = GUI_run_me_anfatec()
% 

close all;
clear all;
clear java;

S.GUI_flag = 1;   % variable to tell Nano6_Flatten_surf_poly that it was called from the GUI.

SCR = get(0,'Screensize');  % Get screensize.
S.fh = figure('numbertitle','off',...
              'menubar','none',...
              'units','pixels',...
              'position',[SCR(3)/2-400 ,SCR(4)/2-300 , 1500, 750],...
              'name','Automated AFM Image Processing',...
              'resize','on');
          
S.plots = uipanel('BackgroundColor','white', 'units', 'pixel', 'Position', [15 150 1170 530], 'visible', 'on');
% For the image:
S.ax1 = axes('units','pixels', 'position',[20 100 500 400], 'Parent', S.plots, 'visible', 'off');
% For the histogram:
S.ax2 = axes('units','pixels', 'position',[630 100 460 400], 'Parent', S.plots, 'visible', 'off');

S.initialText = uicontrol('style', 'text', 'units','pixels', 'position',[480 300 250 200], ...
                    'string',{'- Select the starting directory', '- Select files to process', '- Select the channels to process', ...
                    '- Choose the parameters', '- and Click "Process single image"'},...
                    'BackgroundColor', 'white', 'fontsize',11, 'fontangle', 'italic', 'HorizontalAlignment', 'left', 'visible', 'on');


% Create GUI Text
S.hText = uicontrol('Style','text',...
                  'Units','pixels',...
                  'Position',[1200 150 275 525]);
S.hSlider = uicontrol('Style','slider',...
                    'Units','pixels',...
                    'Position',[1475 150 10 525],...
                    'Enable','off',...
                    'Callback',{@callback_scrolltext,S.hText});

%%Generate some Initial Text
update_scrolltext({'Flatting Progress Display:'},S.hText,S.hSlider);


% Sets the Color bar to display
S.ColorBarText          = uicontrol('Style', 'text', 'String', 'Colorbar: ',...
                                    'Position', [350 20  70 20], 'Fontsize', 11, 'BackgroundColor','white', 'Parent', S.plots);
S.ColorBarSelect        = uicontrol('Style', 'popup',...                  % Channel selection for the plotting                    
                                    'String', { 'Sky',...
                                                'Autumn',...
                                                'Bone',...
                                                'Cool',...
                                                'Copper',...
                                                'Flag',...
                                                'Gray',...
                                                'Hot',...
                                                'HSV',...
                                                'Jet',...
                                                'Pink',...
                                                'Spring',...
                                                'Summer',...
                                                'Winter'},...
                                    'Position', [425 22 150 20], 'fontsize', 11, 'Parent', S.plots);
                
                
                
% For channel display selection:
S.channelDisplayText    = uicontrol('Style', 'text', 'String', 'Display: ',...
                                    'Position', [600 20 70 20], 'Fontsize', 11, 'BackgroundColor','white', 'Parent', S.plots);
S.channelDisplaySelect  = uicontrol('Style', 'popup',...                  % Channel selection for the plotting                    
                                    'String', {'Height', 'Deflection'}',...
                                    'Position', [675 22 150 20], 'fontsize', 11, 'Parent', S.plots);

% Channel State to plot       
S.j                     = uicontrol('Style', 'text', 'position', [700 700 140 40], 'string', '1', 'visible', 'off'); % text that is never visible; to know which channel we are dealing with
S.image_to_plot         = uicontrol('Style', 'text',...
                                    'position', [700 700 140 40], 'string', '1', 'visible', 'off'); % text that is never visible; to know which image (final, raw, etc) to plot
                                
S.raw_vs_processed      = uicontrol('Style', 'popup',...               
                                    'String', 'Final image and histogram|Raw image and histogram|After 1st median correction|After 1st poly flattening|After 2nd median correction|After 2nd poly flattening',...
                                    'Position', [850 22 300 20], 'fontsize', 11, 'Parent', S.plots, 'enable', 'off');

S.pb(1) = uicontrol(    'style','pushbutton',...
                        'units','pixels',...
                        'position',[970 20 200 40],...
                        'string','Process all images',...
                        'fontsize',12, 'enable', 'off');
              
S.pb(2) = uicontrol(    'style','pushbutton',...
                        'units','pixels',...
                        'position',[720 20 200 40],... 
                        'string','Process single image',...
                        'fontsize',12, 'enable', 'off');
              
S.selectFile            = uicontrol('style','pushbutton', 'units','pixels',...
                                    'position',[720 70 200 25], 'string','Select images to process:', 'fontsize',10);
S.selectFileText        = uicontrol('style', 'edit', 'units', 'pixels',...
                                    'position',[920 70 250 25], 'string','', 'fontsize',10);

S.selectFolder          = uicontrol('style','pushbutton', 'units','pixels',...
                                    'position',[720 105 200 25], 'string','Change starting folder:', 'fontsize',10);
S.selectFolderText      = uicontrol('style', 'edit', 'units', 'pixels',...
                                    'position',[920 105 250 25], 'string','C:\', 'fontsize', 10);

S.channelsSelection     = uipanel(  'BackgroundColor','white', 'title', 'Channels to process', 'units', 'pixel',...
                                    'position', [15 20 680 120], 'fontsize', 12, 'visible', 'on');
for k = 1:8
    strposition = [-53+k*80 40 70 40];
    S.channelProcessText(k) = uicontrol('Style','text', 'string', {'Channel ', num2str(k)}, 'position', strposition, 'visible', 'on', 'HorizontalAlignment', 'center','BackgroundColor','white', 'fontsize',10, 'Parent', S.channelsSelection);
end                                                 
for k = 1:8
    strposition = [-25+k*80 25 30 20];
    if k == 1
        S.channelProcessSelect(k) = uicontrol('Style','Check', 'position', strposition, 'Value', 1, 'visible', 'on', 'BackgroundColor','white', 'Parent', S.channelsSelection);
    else
        S.channelProcessSelect(k) = uicontrol('Style','Check', 'position', strposition, 'Value', 0, 'visible', 'on', 'BackgroundColor','white', 'Parent', S.channelsSelection);
    end
end

S.slider_z_reference = uicontrol('style','slide',...
                 'unit','pix',...
                 'position',[130 15 150 20],...      
                 'min', -1,'max',1,...
                 'value', 0, ...
                 'parent', S.plots, 'enable', 'off', 'visible', 'on');  


% Channels explanation panel
S.channel_info_Units = [    '[nm]     ';...
                            '[nm]     ';...
                            '[mV]     ';...
                            '[mv]     ';...
                            '[mV]     ';...
                            '[mV]     ';...
                            '[mV]     ';...
                            '[]      ';...
                            '[mV]     ';...
                            '[mV]     ';...
                            '[mV]     ';...
                            '[um]     ';...
                            '[um]     ';...
                            '[S]      ';...
                            '[A]      '];
channels_titles = ['Topo:              '; ...
                   'fastz:             '; ...
                   'T-B:               '; ... 
                   'fastTB:            '; ...
                   'L-R:               '; ...
                   'Amplitude:         '; ...
                   'FastR:             '; ...
                   'Phase:             '; ...
                   'LiaX:              '; ...
                   'LiaY:              '; ...
                   'bias:              '; ...
                   'x-direction:       '; ...
                   'y-direction:       '; ...
                   'Zeit:              '; ...
                   'minmax:            '];
channels_titles_array = [channels_titles,S.channel_info_Units];

S.channels_explanation_panel = uipanel('Title','Units','FontSize',12.5, 'units', 'pixel', 'Position', [20 35 190 275], 'visible', 'off');
for k = 1:size(S.channel_info_Units,1)
    strposition = [10 250*(size(S.channel_info_Units,1)-k-1)/(size(S.channel_info_Units,1)+1)+20 140 20];
    S.channels_explanation(k) = uicontrol('Style','text', 'string', channels_titles_array(k,:), 'fontsize', 12, 'position', strposition, 'Parent', S.channels_explanation_panel, 'horizontalAlignment', 'left', 'visible', 'on');
end
% Text in "About" tab
S.tx = uicontrol('style','text',...
                 'units','pixels',...
                 'position',[40 20 1020 660],...
                 'visible','off',...
                 'string',{' ','This is a', 'first attempt to',...
                 'create a GUI','with Matlab.',' ','Copyright:',...
                 'LBNI, EPFL','2011'},...
                 'Fontsize',12,'fontweight','bold');
             
             
%Text in "Help" tab
S.explanations = uicontrol('style','text',...
                 'units','pixels',...
                 'position',[40 20 1020 660],...
                 'visible','off',...
                 'string',{' ','This is a', 'tab to write',...
                 'some explanations'},...
                 'Fontsize',12,'horizontalAlignment','left');
             
               
% Parameters per channel panel
S.parameters_panel = uipanel('Title','Parameters per channel','FontSize',12, 'BackgroundColor','white', 'units', 'pixel', 'Position', [230 35 950 645], 'visible', 'off');
S.axText = axes('units','pixels', 'position',[30 530 900 50], 'Parent', S.parameters_panel, 'XColor', 'w', 'YColor', 'w',  'visible', 'on');


Med_corr_1_init_values                              = ['Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y'];
Med_corr_2_init_values                              = ['Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y','Y'];
vert_med_corr_init_values                           = ['N','N','N','N','N','N','N','N','N','N','N','N','N','N','N','N'];
poly_order_horiz_init_values                        = [ 3,  2,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1 ]+ones(1,length(vert_med_corr_init_values));     % +1 because index 0 doesn't exist in matlab;
poly_order_vert_init_values                         = [ 3,  2,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1 ]+ones(1,length(vert_med_corr_init_values));     % +1 because index 0 doesn't exist in matlab;
peak_fit_values                                     = [ 2,  2,  1,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1 ]+ones(1,length(vert_med_corr_init_values));
peak_shift_values                                   = [ 2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1 ]+ones(1,length(vert_med_corr_init_values));
peak_flatten_values                                 = [ 2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1 ]+ones(1,length(vert_med_corr_init_values));
min_peak_dist_initial_values                        = 25*ones(1,length(vert_med_corr_init_values));
min_peak_height_initial_values                      = 0.3*ones(1,length(vert_med_corr_init_values));
min_std_initial_values                              = [ 5.0e-2,  5.0e-2,  2.0e+0,   2.0e-1,   5.0e-2,   1.0e+1,   5.0e-3,   1.0e-3,  1.0e-2,   1.0e-2,   1.0e-2,   1.0e-2,   1.0e-2,   1.0e-2,   1.0e-2,  1.0e-2];
max_std_initial_values                              = [ 5.0e-1,  5.0e-1,  2.0e+1,   2.0e+0,   1.0e-0,   4.0e+1,   2.0e-2,   5.0e-2,  1.0e-1,   1.0e-1,   1.0e-1,   1.0e-1,   1.0e-1,   1.0e-1,   1.0e-1,  1.0e-1];
z_ref_values                                        = [ 0.6e-0,  0.6e-0,  0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,  0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,   0.0e-0,  0.0e-0];
min_z_values                                        = [-2.0e+0, -2.0e+0, -3.0e+1,  -5.0e+0,  -1.0e+1,  -1.0e+2,  -1.0e-2,  -1.0e-1, -1.0e-0,  -1.0e-0,  -1.0e-0,  -1.0e-0,  -1.0e-0,  -1.0e-0,  -1.0e2,  -1.0e2];
max_z_values                                        = [ 2.0e+0,  2.0e+0,  3.0e+1,   5.0e+0,   1.0e+1,   1.0e+2,   3.0e-2,   1.0e-1,  1.0e-0,   1.0e-0,   1.0e-0,   1.0e-0,   1.0e-0,   1.0e-0,   1.0e2,   1.0e2];


col_spacing = (900-180)/size(S.channel_info_Units,1);
row_spacing = (530-10)/(18);
for k = 1:size(S.channel_info_Units,1)
   for l = 0:17
        % - Titles - %        
        text((160+k*col_spacing)/900, 0.1, channels_titles(k,:),'Fontsize', 12,'HorizontalAlignment', 'Left','Rotation',90);                                                                 
        
        
        % - Median correction - %
        if l == 17
        S.Med_corr_text                             = uicontrol(    'Style', 'text', 'String', 'Median correction:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'HorizontalAlignment', 'left',...
                                                                    'Visible', 'on');                                                                  
            strposition                             = [180+k*col_spacing 10+l*row_spacing 30 20];
            if (Med_corr_1_init_values(k) == 'Y')
                S.Med_corr(k)                       = uicontrol(    'Style','Check',... 
                                                                    'Position', strposition, 'Value', 1,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'Visible', 'on');
            else
                S.Med_corr(k)                       = uicontrol(    'Style','Check',...
                                                                    'Position', strposition, 'Value', 0,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'Visible', 'on');
            end
        end
        
        % - Median correction 2 - %
        if l == 16
        S.Med_corr_2_text                           = uicontrol(    'Style', 'text', 'String', 'Median correction 2:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'HorizontalAlignment', 'left',...
                                                                    'Visible', 'on');

            strposition                             = [180+k*col_spacing 10+l*row_spacing 30 20];
            if (Med_corr_2_init_values(k) == 'Y')
                S.Med_corr_2(k)                     = uicontrol(    'Style','Check',...
                                                                    'Position', strposition, 'value', 1,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'Visible', 'on');
            else
                S.Med_corr_2(k)                     = uicontrol(    'Style','Check',...
                                                                    'Position', strposition, 'value', 0,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'Visible', 'on');
            end
        
        end
        
        % - Vertical median correction - %
        if l == 15
        S.vert_med_corr_text                        = uicontrol(    'Style', 'text', 'String', 'Vertical median correction:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [180+k*col_spacing 10+l*row_spacing 30 20];
            if (vert_med_corr_init_values(k) == 'Y')
                S.vert_med_corr(k)                  = uicontrol(    'Style','Check','position', strposition,...
                                                                    'value', 1, 'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
            else
                S.vert_med_corr(k)                  = uicontrol(    'Style','Check','position', strposition,...
                                                                    'value', 0, 'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
            end
        
        end
        
        % - Polynomial flattening order, horizontal - %
        if l == 14
        S.poly_order_horiz_text                     = uicontrol(    'Style', 'text', 'String', 'Poly order horiz:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.poly_order_horiz(k)                   = uicontrol(     'Style','popup','String','0|1|2|3|4|5',...
                                                                     'position', strposition, 'Value', poly_order_horiz_init_values(k),...
                                                                     'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                     'visible', 'on');
        
        end                                                         
        % - Polynomial flattening order, vertical - %
        if l == 13
        S.poly_order_vert_text                      = uicontrol(    'Style', 'text', 'String', 'Poly order vert:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.poly_order_vert(k)                    = uicontrol(    'Style','popup','String','0|1|2|3|4|5',...
                                                                    'position', strposition, 'value', poly_order_vert_init_values(k),...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % % - Min Peak Distance - %
        if l == 12
        S.min_peak_distance_reference_text          = uicontrol(    'Style', 'text', 'String', 'Minimum Peak Distance (in units):',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.min_peak_distance(k)                  = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', min_peak_dist_initial_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        
        end
        
        % % - Min Peak Height - %
        if l == 11
        S.min_peak_height_reference_text            = uicontrol(    'Style', 'text', 'String', 'Minimum Peak Height (%):',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.min_peak_height(k)                    = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', min_peak_height_initial_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        
        end
        
        % - min_spacing_stdev_user - %
        if l == 10
        S.min_spacing_stdev_user_text               = uicontrol(    'Style', 'text', 'String', 'Min spacing standard deviation:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.min_spacing_stdev_user(k)             = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', min_std_initial_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - max_spacing_stdev_user - %
        if l == 9
        S.max_spacing_stdev_user_text               = uicontrol(    'Style', 'text', 'String', 'Max spacing standard deviation:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.max_spacing_stdev_user(k)             = uicontrol(    'Style','edit','position', strposition,....
                                                                    'Parent', S.parameters_panel, 'string', max_std_initial_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        
        end
        
        % - calc_flatten_lim - %
        if l == 8
        S.Calc_flatten_lim_text                     = uicontrol (   'Style', 'text', 'String', 'Calc. flatten limits:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [180+k*col_spacing 10+l*row_spacing 30 20];
            S.Calc_flatten_lim(k)                   = uicontrol(    'Style','Check','position', strposition, 'value', 1,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - overwrite_Z lim -%
        if l == 7
        S.overwrite_xlim_text                       = uicontrol (   'Style', 'text', 'String', 'Overwrite Z Limits:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
             strposition                            = [180+k*col_spacing 10+l*row_spacing 30 20];
             S.overwrite_xlim(k)                    = uicontrol(    'Style','Check','position', strposition, 'value', 1,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % % - Z Reference Value - %
        if l == 6
        S.z_height_reference_text                   = uicontrol(    'Style', 'text', 'String', 'z height reference:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.z_height_reference(k)                 = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', z_ref_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');

        end
        
        % - min_z - %
        if l == 5
        S.min_z_text                                = uicontrol(    'Style', 'text', 'String', 'Minimum z:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.min_z(k)                              = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', min_z_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - max_z - %
        if l == 4
        S.max_z_text                                = uicontrol(    'Style', 'text', 'String', 'Maximum z:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'HorizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.max_z(k)                              = uicontrol(    'Style','edit','position', strposition,...
                                                                    'Parent', S.parameters_panel, 'string', max_z_values(k), 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - peaks_fit - %
        if l == 3
        S.peaks_fit_text                            = uicontrol(    'Style', 'text', 'String', 'Number of peaks to fit:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'HorizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.peaks_fit(k)                          = uicontrol(    'Style','popup','String','0|1|2|3|4|5',...
                                                                    'Position', strposition, 'value', peak_fit_values(k),...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - peak_reference_height - %
        if l == 2
        S.peak_reference_height_text                = uicontrol(    'Style', 'text', 'String', 'Peak to shift:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.peak_reference_height(k)              = uicontrol(    'Style','popup','String','0|1|2|3|4|5',...
                                                                    'Position', strposition, 'value', peak_shift_values(k),...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        
        % - sort_peaks - %
        if l == 1
        S.sort_peaks_text                           = uicontrol (   'Style', 'text', 'String', 'Sort peaks in position order:',...
                                                                    'Position', [15 10+l*row_spacing 200 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','White',...
                                                                    'HorizontalAlignment', 'Left', 'Visible', 'on');
            strposition                             = [180+k*col_spacing 10+l*row_spacing 30 20];
            S.sort_peaks(k)                         = uicontrol(    'Style','Check','position', strposition, 'value', 1,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        end
        % - peak_to_flatten - %
        if l == 0
        S.peak_to_flatten_text                      = uicontrol(    'Style', 'text', 'String', 'Peak to flatten (position order):',...
                                                                    'Position', [15 10+l*row_spacing 180 20], 'Fontsize', 12,...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'horizontalAlignment', 'left', 'visible', 'on');
            strposition                             = [175+k*col_spacing 10+l*row_spacing 30 20];
            S.peak_to_flatten(k)                    = uicontrol(    'Style','popup','String','0|1|2|3|4|5',...
                                                                    'Position', strposition, 'value', peak_flatten_values(k),...
                                                                    'Parent', S.parameters_panel, 'BackgroundColor','white',...
                                                                    'visible', 'on');
        
        end

   end
end










S.other_parameters_panel    = uipanel('Title','Other parameters','FontSize',12, 'BackgroundColor','white', 'units', 'pixel', 'Position', [20 400 190 280], 'visible', 'off');
S.disp_scale_bar_text       = uicontrol('Style','text','string', 'Scale Bar:',                          'Position',[10  0/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.disp_scale_bar            = uicontrol('Style','Check',                                                'Position',[120 0/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','Value', 1);
S.disp_grain_analysis_text  = uicontrol('Style','text','string', 'Grain Analysis:',                     'Position',[10  1/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.disp_grain_analysis       = uicontrol('Style','Check',                                                'Position',[120 1/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','Value', 0);
S.scale_bar_length_text     = uicontrol('Style','text','string', 'Scale Bar (nm):',                     'Position',[10  2/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.scale_bar_length          = uicontrol('Style','edit',                                                 'Position',[120 2/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 1000);
S.crop_fraction_text        = uicontrol('Style','text','string', 'Crop Fraction:',                      'Position',[10  3/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.crop_fraction             = uicontrol('Style','edit',                                                 'Position',[120 3/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 0);
S.thresh_range_text         = uicontrol('Style','text','string', 'Threshold Range:',                    'Position',[10  4/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.thresh_range              = uicontrol('Style','edit',                                                 'Position',[120 4/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 5);
S.peak_threshold_text       = uicontrol('Style','text','string', 'Peak Threshold:',                     'Position',[10  5/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.peak_threshold            = uicontrol('Style','edit',                                                 'Position',[120 5/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 0);
S.smoothing_level_text      = uicontrol('Style','text','string', 'Smoothing Width:',                    'Position',[10  6/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.smoothing_level           = uicontrol('Style','edit',                                                 'Position',[120 6/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 8);
S.histogram_bins_text       = uicontrol('Style','text','string', 'Histogram Bins:',                     'Position',[10  7/8*250+10 100 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on');
S.histogram_bins            = uicontrol('Style','edit',                                                 'Position',[120 7/8*250+10  40 20], 'Parent', S.other_parameters_panel, 'BackgroundColor', 'white', 'Fontsize', 12, 'HorizontalAlignment', 'Left','Visible', 'on','string', 256 );



S.save = uicontrol('style','pushbutton','units','pixels', 'position',[55 355 120 30], 'string','Save parameters','Fontsize', 12, 'visible', 'off');
S.load = uicontrol('style','pushbutton','units','pixels', 'position',[55 315 120 30], 'string','Load parameters','Fontsize', 12, 'visible', 'off');

% Toggles will act as the tabs.              
S.tg(1) = uicontrol('style','toggle',...
                    'units','pixels',...
                    'position',[5 700 140 40],...
                    'string','MAIN','fontweight', 'bold',...
                    'fontsize', 11, ...
                    'val',1);
S.tg(2) = uicontrol('style','toggle',...
                    'units','pixels',...
                    'position',[145 700 140 40],...
                    'string','PARAMETERS','fontweight', 'bold',...
                    'fontsize', 11, ...
                    'value',0,...
                    'enable','on');
S.tg(3) = uicontrol('style','toggle',...
                    'units','pixels',...
                    'position',[285 700 120 40],...
                    'string','HELP', 'fontweight', 'bold',...
                    'fontsize', 11, ...
                    'value',0,...
                    'enable','on');
S.tg(4) = uicontrol('style','toggle',...
                    'units','pixels',...
                    'position',[405 700 120 40],...
                    'string','ABOUT', 'fontweight', 'bold',...
                    'fontsize', 11, ...
                    'value',0,...
                    'enable','on');
                              
S.text_waiting = uicontrol('style', 'text', 'units', 'pixels', 'position', [17 155 1160 515], 'string', 'Processing...', 'horizontalAlignment', 'center', 'BackgroundColor', 'white', 'fontsize', 14, 'visible', 'off');

S.slider_val = uicontrol('style', 'text', 'position', [130 42 150 20], 'string', get(S.z_height_reference(1), 'string'), 'Fontsize', 12, 'backgroundcolor', 'white', 'parent', S.plots, 'visible', 'off');  
S.slider_min = uicontrol('style', 'edit', 'position', [75 15 50 20],   'string', get(S.min_z(1), 'string'),              'Fontsize', 12, 'backgroundcolor', 'white', 'parent', S.plots, 'visible', 'off');  
S.slider_max = uicontrol('style', 'edit', 'position', [285 15 50 20],  'string', get(S.max_z(1), 'string'),              'Fontsize', 12, 'backgroundcolor', 'white', 'parent', S.plots, 'visible', 'off');  

S.Offset_data = 0;

% -- Set the callback functions. --%
set(S.pb(1),'callback',{@processAllImages_call,S}) 
set(S.pb(2),'callback',{@processOneImage_call,S})  
set(S.tg(:),{'callback'},{{@tg_call,S}})
set(S.raw_vs_processed,'callback',{@plots_call,S});
set(S.ColorBarSelect,'callback',{@colorbar_call,S});  
set(S.channelDisplaySelect, 'callback', {@display_channels_call,S}); 
set(S.save, 'callback', {@save_call,S}); 
set(S.load, 'callback', {@load_call,S}); 
set(S.selectFolder, 'callback', {@selectFolder_call,S});
set(S.selectFile, 'callback', {@selectFile_call,S});
set(S.slider_z_reference, 'callback', {@slider_z_call,S});
set([S.slider_min; S.slider_max], 'callback', {@slider_min_max_call,S});
set([S.min_z(:); S.max_z(:); S.z_height_reference(:)] , 'callback', {@change_slider_z_call,S});
set(S.channelProcessSelect(:), 'callback', {@selectChannels_call,S});


%% Select the Folder with Data Files
function [] = selectFolder_call(varargin)
[~,S] = varargin{[1,3]};  % Get the structure.
S.folder = uigetdir;
set(S.selectFolderText, 'string', S.folder);

%% Select the Files to Process
function [] = selectFile_call(varargin)
[~,S] = varargin{[1,3]};  % Get the structure.
[S.Filename, S.ImageDir] = uigetfile({'*.txt','Anfatec Data Files (*.txt)'}, 'Pick files', get(S.selectFolderText, 'string'),...
                                     'MultiSelect','on') ; 
assignin('base', 'Filename', S.Filename);
set(S.selectFileText, 'string', [S.Filename{1}, ', ...']);
channelsInfo = getChannelInfo_anfatec(S.Filename, S.ImageDir);
for k = 1:min(size(channelsInfo, 1),8)
    set(S.channelProcessText(k), 'string', {channelsInfo(k, 1).Name});
end
set(S.pb(2), 'enable', 'on');   % Enables the "Process single image" button now that the file to process has been selected
selectChannels_call(varargin);   % Disable all the parameters of channels that are not processed

%% Select Channels to Process
function [] = selectChannels_call(varargin)
if size(varargin,2) == 1 % callback was called from processOneImage_call
    varargin = varargin{1};
    [~,S] = varargin{[1,3]};  % Get the structure.
else
    [~,S] = varargin{[1,3]};  % Get the structure.
end

[display_nb ~] = find(cell2mat(get(S.channelProcessSelect, 'value'))==1); 
channels = get(S.channelProcessText(display_nb), 'string');
if (length(channels)>1)
    channels = [channels{1:length(channels)}];
else
    channels = channels;
end

enabled_channels = zeros(1, 15);
for i = 1:size(channels,2)
    switch channels{i}                                                                                                                                                   
        case 'Topo'                                                                                                                                                        
             enabled_channels(1) = 1;
        case 'fastz'                                                                                                                                                        
             enabled_channels(2) = 1;
        case 'T-B'                                                                                                                                                          
             enabled_channels(3) = 1;
        case 'fastTB'                                                                                      
             enabled_channels(4) = 1;
        case 'L-R'                                                                                                                                                          
             enabled_channels(5) = 1;
        case 'Amplitude'                                                                                                                                         
             enabled_channels(6) = 1;
        case 'FastR'   
             enabled_channels(7) = 1;
        case 'Phase'                                            
             enabled_channels(8) = 1;
        case 'LiaX'                                                                                             
             enabled_channels(9) = 1;
        case 'LiaY'                                                                    
             enabled_channels(10) = 1;
        case 'bias'                                                               
             enabled_channels(11) = 1;
        case 'x-direction'                                                                
             enabled_channels(12) = 1;
        case 'y-direction'                                                                                
             enabled_channels(13) = 1;
        case 'Zeit'                                                                                                                                         
             enabled_channels(14) = 1;
        case 'minmax'                                                                                                              
             enabled_channels(15) = 1;
    end
end

for k = 1:size(S.channel_info_Units,1)
    if enabled_channels(k) == 0    % channels that do not exist and the ones that are not selected = disabled channels
        set(S.Med_corr(k), 'enable', 'off');
        set(S.Med_corr_2(k),'enable', 'off');
        set(S.vert_med_corr(k), 'enable', 'off');
        set(S.poly_order_horiz(k), 'enable', 'off');
        set(S.poly_order_vert(k), 'enable', 'off');
        set(S.z_height_reference(k), 'enable', 'off');
        set(S.min_spacing_stdev_user(k), 'enable', 'off');
        set(S.max_spacing_stdev_user(k), 'enable', 'off');
        set(S.min_z(k), 'enable', 'off');
        set(S.max_z(k), 'enable', 'off');
        set(S.peaks_fit(k), 'enable', 'off');
        set(S.peak_reference_height(k), 'enable', 'off');
        set(S.peak_to_flatten(k), 'enable', 'off');
        set(S.Calc_flatten_lim(k), 'enable', 'off');
        set(S.overwrite_xlim(k), 'enable', 'off');
        set(S.sort_peaks(k), 'enable', 'off');
    else  % enabled channels
        set(S.Med_corr(k), 'enable', 'on');
        set(S.Med_corr_2(k),'enable', 'on');
        set(S.vert_med_corr(k), 'enable', 'on');
        set(S.poly_order_horiz(k), 'enable', 'on');
        set(S.poly_order_vert(k), 'enable', 'on');
        set(S.z_height_reference(k), 'enable', 'on');
        set(S.min_spacing_stdev_user(k), 'enable', 'on');
        set(S.max_spacing_stdev_user(k), 'enable', 'on');
        set(S.min_z(k), 'enable', 'on');
        set(S.max_z(k), 'enable', 'on');
        set(S.peaks_fit(k), 'enable', 'on');
        set(S.peak_reference_height(k), 'enable', 'on');
        set(S.peak_to_flatten(k), 'enable', 'on');
        set(S.Calc_flatten_lim(k), 'enable', 'on');
        set(S.overwrite_xlim(k), 'enable', 'on');
        set(S.sort_peaks(k), 'enable', 'on');
    end
end

%% Process the First Image
function [] = processOneImage_call(varargin)
% Callback for pushbutton "Process one image"
[~,S] = varargin{[1,3]};
delete([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']);
%set(S.text_waiting, 'visible', 'on'); 
set(S.initialText, 'visible', 'off');
% set(varargin{3}.tg(2:4),'enable','off');  % Turn off the other tabs.
set(varargin{3}.pb(1:2),'enable', 'off'); % Turn off the "Process one image" and "Process all images" buttons
S.processAll = 0; % To tell Nano6_Flatten_surf_poly that it has to process only one image
update_scrolltext([],S.hText,S.hSlider);
pause(0.0001);
Nano6_Flatten_surf_poly_GUI_anfatec;     
set(varargin{3}.tg(2:4),'enable','on');  % Turn on the other tabs.
set(varargin{3}.pb(1:2),'enable', 'on'); % Turn on the "Process one image" and "Process all images" buttons
set(S.raw_vs_processed, 'enable', 'on');
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly
display_channels_call(varargin)
j = str2double(get(S.j, 'string'));
set(S.slider_z_reference, 'min', str2double(get(S.min_z(j), 'string')));
set(S.slider_z_reference, 'max', str2double(get(S.max_z(j), 'string')));
set(S.slider_z_reference, 'value', str2double(get(S.z_height_reference(j), 'string')));
set(S.slider_val, 'string', [get(S.z_height_reference(j), 'string'), ' ', S.channel_info_Units(j,:)]);
set(S.slider_min, 'string', get(S.min_z(j), 'string'));
set(S.slider_max, 'string', get(S.max_z(j), 'string'));
set(S.channelDisplaySelect, 'enable', 'on');
images(:,:,j) = final_images(:,:,j);
S.Offset_data = 0;
plot_nice_image_in_GUI_anfatec;
%set(S.text_waiting, 'visible', 'off');
[display_nb ~] = find(cell2mat(get(S.channelProcessSelect, 'value'))==1); 
disp_str = get(S.channelProcessText(display_nb), 'string');
set(S.channelDisplaySelect, 'string', [disp_str{1:length(disp_str)}]);
set(S.channelDisplaySelect, 'value', 1);
set(S.slider_z_reference, 'enable', 'on');
set([S.slider_z_reference; S.slider_val; S.slider_min; S.slider_max], 'visible', 'on');

%% Process All Images
function [] = processAllImages_call(varargin)
% Callback for pushbutton "Process all images"
[~,S]                                       = varargin{[1,3]};
set(varargin{3}.tg(2:4),'enable','off');  % Turn off the other tabs.
set(varargin{3}.pb(1:2),'enable', 'off'); % Turn off the "Process one image" and "Process all images" buttons
set(S.raw_vs_processed, 'enable', 'off')
set(S.channelDisplaySelect, 'enable', 'off');
S.processAll                                = 1; % Variable to tell Nano6_Flatten_surf_poly that it has to process only all images (if processAll = 1), or only one image (if processAll = 0)
S.Offset_data                               = 0;
update_scrolltext([],S.hText,S.hSlider);
pause(0.0001);
Nano6_Flatten_surf_poly_GUI_anfatec;
set(varargin{3}.tg(2:4),'enable','on');  % Turn on the other tabs.
set(varargin{3}.pb(1:2),'enable', 'on'); % Turn on the "Process one image" and "Process all images" buttons

%% Control the Vertical Display Props
function [] = slider_z_call(varargin)
% Callback to change z_reference_height with slider; can only be called
% when image to display is the final image, because slider is disabled else
[~,S] = varargin{[1,3]};  % Get the structure.
j = str2double(get(S.j, 'string')); % j = channel
set(S.z_height_reference(j), 'string', num2str(get(S.slider_z_reference, 'value')));
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
set(S.slider_val, 'string', [get(S.z_height_reference(j), 'string'), ' ', S.channel_info_Units(j,:)]);
S.Offset_data = 0;
images(:,:,j) = final_images(:,:,j);
plot_nice_image_in_GUI_anfatec; % to refresh the image with the new offset
S.Offset_data = 0;
function [] = slider_min_max_call(varargin)
% Callback for when min_z or max_z was changed (limits of the slider)
[~,S] = varargin{[1,3]};  % Get the structure.
j = str2double(get(S.j, 'string')); % j = channel
set(S.slider_z_reference, 'value', (str2double(get(S.slider_min, 'string'))+str2double(get(S.slider_max, 'string')))/2)
set(S.slider_z_reference, 'min', str2double(get(S.slider_min, 'string')));
set(S.min_z(j), 'string', get(S.slider_min, 'string'));
set(S.slider_z_reference, 'max', str2double(get(S.slider_max, 'string')));
set(S.max_z(j), 'string', get(S.slider_max, 'string'));
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
images(:,:,j) = final_images(:,:,j);
plot_nice_image_in_GUI_anfatec; % to refresh the image with the new offset

function [] = change_slider_z_call(varargin)
% Callback for changing the min, max and current values of the slider
[~,S] = varargin{[1,3]};  % Get the structure.
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
j = str2double(get(S.j, 'string')); % j = channel
set(S.slider_z_reference, 'min', str2double(get(S.min_z(j), 'string')));
set(S.slider_z_reference, 'max', str2double(get(S.max_z(j), 'string')));
set(S.slider_z_reference, 'value', str2double(get(S.z_height_reference(j), 'string')));
set(S.slider_val, 'string', [get(S.z_height_reference(j), 'string'), ' ', S.channel_info_Units(j,:)]);
set(S.slider_min, 'string', get(S.min_z(j), 'string'));
set(S.slider_max, 'string', get(S.max_z(j), 'string'));
images(:,:,j) = final_images(:,:,j);
plot_nice_image_in_GUI_anfatec;

function [] = display_channels_call(varargin)
% Callback for the popup allowing the choice of which channel to display
if size(varargin,2) == 1 % callback was called from processOneImage_call
    varargin = varargin{1};
    [~,S] = varargin{[1,3]};  % Get the structure.
    for k = 1:7
        if get(S.channelProcessSelect(k), 'value') == 1
           channel_name = get(S.channelProcessText(k), 'string');
           channel_name = channel_name{1};
           break;
        end
    end
else
    [~,S] = varargin{[1,3]};  % Get the structure.
    channel_string = get(S.channelDisplaySelect, 'string');
    channel_name = channel_string(get(S.channelDisplaySelect, 'Value'));
    channel_name = channel_name{1};
end
        switch channel_name                                                                                                                                                
                case 'Topo'                                                                                                                                                        
                     set(S.j, 'string', '1');
                case 'fastz'                                                                                                              
                     set(S.j, 'string', '2');
                case 'T-B'                                                                                                                                                          
                     set(S.j, 'string', '3');
                case 'fastTB'                                                                                      
                     set(S.j, 'string', '4');
                case 'L-R'                                                                                                                                                         
                     set(S.j, 'string', '5');
                case 'Amplitude'                                                                                                                                        
                     set(S.j, 'string', '6');
                case 'FastR'   
                     set(S.j, 'string', '7');
                case 'Phase'                                            
                     set(S.j, 'string', '8');
                case 'LiaX'                                                                                             
                     set(S.j, 'string', '9');
                case 'LiaY'                                                                    
                     set(S.j, 'string', '10');
                case 'bias'                                                               
                     set(S.j, 'string', '11');
                case 'x-direction'                                                                
                     set(S.j, 'string', '12');
                case 'y-direction'                                                                                
                     set(S.j, 'string', '13');
                case 'Zeit'                                                                                                                                         
                     set(S.j, 'string', '14');
                case 'minmax'                                                                                                              
                     set(S.j, 'string', '15');
        end
        
j = str2double(get(S.j, 'string'));
set(S.image_to_plot, 'string', '1') % if we change the channel to display, always displays first the final image
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
set(S.slider_z_reference, 'min', str2double(get(S.min_z(j), 'string'))); % Refresh the slider's parameters
set(S.slider_z_reference, 'max', str2double(get(S.max_z(j), 'string')));
set(S.slider_z_reference, 'value', str2double(get(S.z_height_reference(j), 'string')));
set(S.slider_val, 'string', [get(S.z_height_reference(j), 'string'), ' ', S.channel_info_Units(j,:)]);
set(S.slider_min, 'string', get(S.min_z(j), 'string'));
set(S.slider_max, 'string', get(S.max_z(j), 'string'));
popup_items = [1, 1, get(S.Med_corr(j), 'value'), 1,1, get(S.Med_corr_2(j), 'value'), 1];
images_str = {'Final image and histogram'; 'Raw image and histogram'; 'After 1st median correction';'After General Tilt Correction'; 'After 1st poly flattening'; 'After 2nd median correction'; 'After 2nd poly flattening'};
set(S.raw_vs_processed, 'string', images_str(popup_items == 1));
set(S.raw_vs_processed,'value', 1);
images(:,:,j) = final_images(:,:,j);
set(S.slider_z_reference, 'enable', 'on');
plot_nice_image_in_GUI_anfatec;

function [] = plots_call(varargin)
% Callback for the popup "Display raw image and histogram|Display processed image and histogram".
[~,S] = varargin{[1,3]};  % Get the structure.
set(S.image_to_plot, 'string', num2str(get(S.raw_vs_processed,'value')));  % Get the user's choice.
j = str2double(get(S.j, 'string')); % j = channel
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
%string = images_str(str2double(get(S.image_to_plot, 'string')));
all_strings = get(S.raw_vs_processed, 'string');
value       = get(S.raw_vs_processed, 'value');
switch all_strings{value}   %string{1}
    case 'Final image and histogram'
        images(:,:,j) = final_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'on');
        set(S.image_to_plot, 'string', '1');
    case 'Raw image and histogram'
        images(:,:,j) = raw_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '2');
    case 'After 1st median correction'
        images(:,:,j) = med_corr_1_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '3');
    case 'After General Tilt Correction'
        images(:,:,j) = ini_tilt_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '4');
    case 'After 1st poly flattening'
        images(:,:,j) = poly_flat_1_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '5');
    case 'After 2nd median correction'
        images(:,:,j) = med_corr_2_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '6');
    case 'After 2nd poly flattening'
        images(:,:,j) = poly_flat_2_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');  
        set(S.image_to_plot, 'string', '7');
end
plot_nice_image_in_GUI_anfatec;

function [] = colorbar_call(varargin)
% Callback for the popup "Display raw image and histogram|Display processed image and histogram".
[~,S] = varargin{[1,3]};  % Get the structure.
set(S.image_to_plot, 'string', num2str(get(S.raw_vs_processed,'value')));  % Get the user's choice.
j = str2double(get(S.j, 'string')); % j = channel
load([get(S.selectFolderText, 'string'),filesep,'Nano6_variables.mat']) % Loads the variables used in Nano6_Flatten_surf_poly to be able to plot the image
%string = images_str(str2double(get(S.image_to_plot, 'string')));
all_strings = get(S.raw_vs_processed, 'string');
value       = get(S.raw_vs_processed, 'value');
switch all_strings{value}   %string{1}
    case 'Final image and histogram'
        images(:,:,j) = final_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'on');
        set(S.image_to_plot, 'string', '1');
    case 'Raw image and histogram'
        images(:,:,j) = raw_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '2');
    case 'After 1st median correction'
        images(:,:,j) = med_corr_1_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '3');
    case 'After General Tilt Correction'
        images(:,:,j) = ini_tilt_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '4');
    case 'After 1st poly flattening'
        images(:,:,j) = poly_flat_1_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '5');
    case 'After 2nd median correction'
        images(:,:,j) = med_corr_2_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');
        set(S.image_to_plot, 'string', '6');
    case 'After 2nd poly flattening'
        images(:,:,j) = poly_flat_2_images(:,:,j);
        set(S.slider_z_reference, 'enable', 'off');  
        set(S.image_to_plot, 'string', '7');
end
plot_nice_image_in_GUI_anfatec;

function [] = tg_call(varargin)
% Callback for tabs
[h,S] = varargin{[1,3]};  % Get calling handle and structure.

if get(h,'val')==0  % Here the Toggle is already pressed.
    set(h,'val',1) % To keep the Tab-like functioning.
end
% Each case of the switch has one toggle associated with it.  When a toggle
% is selected the uicontrols which belong to it are made visible, and the
% others are made invisible.  This way each toggle has, in effect, its own 
% GUI.  All uicontrols are part of the main GUI, some are simply hidden
% when each toggle is selected.  This mimics the action of tabs.
switch h
    case S.tg(1)
        set(S.tg([2,3,4]),'val',0)   
        set([S.plots;S.pb(:); S.channelsSelection; S.selectFolder; S.selectFile;...
            S.selectFolderText; S.selectFileText],{'visible'},{'on'})
        set([S.tx; S.parameters_panel; S.channels_explanation_panel; S.other_parameters_panel;...
            S.explanations; S.save; S.load; S.initialText],{'visible'},{'off'})
    case S.tg(2)
        set(S.tg([1,3,4]),{'val'},{0})
        set([S.parameters_panel; S.channels_explanation_panel; S.other_parameters_panel; S.save; S.load],'visible','on')
        set([S.plots;S.pb(:);S.tx; S.channelsSelection; S.explanations; S.selectFolder; S.selectFile;...
            S.selectFolderText; S.selectFileText; S.initialText],{'visible'},{'off'})
    case S.tg(3)
        set(S.tg([1,2,4]),{'val'},{0})
        set([S.parameters_panel; S.channels_explanation_panel; S.other_parameters_panel; S.plots;S.pb(:);S.tx; ...
            S.channelsSelection; S.save; S.load; S.selectFolder; S.selectFile; S.selectFolderText; S.selectFileText; S.initialText],'visible','off')
        set([S.explanations],{'visible'},{'on'})
    otherwise
        set(S.tg([1,2,3]),{'val'},{0})
        set(S.tx,'visible','on')
        set([S.plots;S.pb(:); S.channelsSelection; S.parameters_panel; S.channels_explanation_panel; S.other_parameters_panel; ...
            S.explanations; S.save; S.load; S.selectFolder; S.selectFile; S.selectFolderText; S.selectFileText; S.initialText],{'visible'},{'off'}) 
end

%% Save Fitting Settings
function [] = save_call(varargin)
% Callback for the button "Save". Saves the current parameters
%[FileName,PathName] = uiputfile('settings.m','Save as');
S = varargin{3};
histogram_bins                  = get(S.histogram_bins, 'string');
smoothing_level                 = get(S.smoothing_level, 'string');
min_peak_height                 = get(S.min_peak_height, 'string');
min_peak_distance               = get(S.min_peak_distance, 'string');
thresh_range                    = get(S.thresh_range, 'string');
Med_corr                        = zeros(size(S.channel_info_Units,1),1);
Med_corr_2                      = zeros(size(S.channel_info_Units,1),1);
vert_med_corr                   = zeros(size(S.channel_info_Units,1),1);
poly_order_horiz                = zeros(size(S.channel_info_Units,1),1);
poly_order_vert                 = zeros(size(S.channel_info_Units,1),1);
z_height_reference              = zeros(size(S.channel_info_Units,1),1);
min_spacing_stdev_user          = zeros(size(S.channel_info_Units,1),1);
max_spacing_stdev_user          = zeros(size(S.channel_info_Units,1),1);
min_z                           = zeros(size(S.channel_info_Units,1),1);
max_z                           = zeros(size(S.channel_info_Units,1),1);
peaks_fit                       = zeros(size(S.channel_info_Units,1),1);
peak_reference_height           = zeros(size(S.channel_info_Units,1),1);
peak_to_flatten                 = zeros(size(S.channel_info_Units,1),1);
Calc_flatten_lim                = zeros(size(S.channel_info_Units,1),1);
overwrite_xlim                  = zeros(size(S.channel_info_Units,1),1);
sort_peaks                      = zeros(size(S.channel_info_Units,1),1);
for k = 1:size(S.channel_info_Units,1)
    Med_corr(k)                 = get(S.Med_corr(k), 'value');
    Med_corr_2(k)               = get(S.Med_corr_2(k),'value');
    vert_med_corr(k)            = get(S.vert_med_corr(k), 'value');
    poly_order_horiz(k)         = get(S.poly_order_horiz(k), 'value');
    poly_order_vert(k)          = get(S.poly_order_vert(k), 'value');
    z_height_reference(k)       = str2double(get(S.z_height_reference(k), 'string'));
    min_spacing_stdev_user(k)   = str2double(get(S.min_spacing_stdev_user(k), 'string'));
    max_spacing_stdev_user(k)   = str2double(get(S.max_spacing_stdev_user(k), 'string'));
    min_z(k)                    = str2double( get(S.min_z(k), 'string'));
    max_z(k)                    = str2double(get(S.max_z(k), 'string'));
    peaks_fit(k)                = get(S.peaks_fit(k), 'value');
    peak_reference_height(k)    = get(S.peak_reference_height(k), 'value');
    peak_to_flatten(k)          = get(S.peak_to_flatten(k), 'value');
    Calc_flatten_lim(k)         = get(S.Calc_flatten_lim(k), 'value');
    overwrite_xlim(k)           = get(S.overwrite_xlim(k), 'value');
    sort_peaks(k)               = get(S.sort_peaks(k), 'value');
end

uisave({'histogram_bins', 'smoothing_level', 'min_peak_height', 'min_peak_distance', ...
    'thresh_range', 'Med_corr', 'Med_corr_2', 'vert_med_corr',...
    'poly_order_horiz', 'poly_order_vert', ...
    'z_height_reference', 'min_spacing_stdev_user', 'max_spacing_stdev_user', 'min_z', ...
    'max_z', 'peaks_fit', 'peak_reference_height', 'peak_to_flatten', ...
    'Calc_flatten_lim', 'overwrite_xlim', 'sort_peaks'},'Settings.mat');

%% Load Fitting Settings
function [] = load_call(varargin)
% Callback for the button "Load".  Loads the parameters previously saved
% with the button "Save".
[~,S] = varargin{[1,3]};  % Get the structure.
uiopen('Settings.mat');
set(S.histogram_bins, 'string', histogram_bins);
set(S.smoothing_level, 'string', smoothing_level);
set(S.thresh_range, 'string', thresh_range);
for k = 1:size(S.channel_info_Units,1)
   set(S.min_peak_distance(k), 'string', min_peak_distance(k));
   set(S.min_peak_height(k), 'string', min_peak_height(k)); 
   set(S.Med_corr(k), 'value', Med_corr(k));
   set(S.Med_corr_2(k), 'value', Med_corr_2(k));
   set(S.vert_med_corr(k), 'value', vert_med_corr(k));
   set(S.poly_order_horiz(k), 'value', poly_order_horiz(k));
   set(S.poly_order_vert(k), 'value', poly_order_vert(k));
   set(S.z_height_reference(k), 'string', z_height_reference(k));
   set(S.min_spacing_stdev_user(k), 'string', min_spacing_stdev_user(k));
   set(S.max_spacing_stdev_user(k), 'string', max_spacing_stdev_user(k));
   set(S.min_z(k), 'string', min_z(k));
   set(S.max_z(k), 'string', max_z(k));
   set(S.peaks_fit(k), 'value', peaks_fit(k));
   set(S.peak_reference_height(k), 'value', peak_reference_height(k));
   set(S.peak_to_flatten(k), 'value', peak_to_flatten(k));
   set(S.Calc_flatten_lim(k), 'value', Calc_flatten_lim(k));
   set(S.overwrite_xlim(k), 'value', overwrite_xlim(k));
   set(S.sort_peaks(k), 'value', sort_peaks(k));
end

%% Callback for scrolling text
function [] = callback_scrolltext(source,~,hText)
  textString = get(hText,'UserData');
  nLines = numel(textString);
  lineIndex = nLines-round(get(source,'Value'));
  set(hText,'String',textString(lineIndex:nLines));
  
%%  Update Scrolling Text
function update_scrolltext(newText,hText,hSlider)
  newText = textwrap(hText,newText);
  set(hText,'String',newText,'UserData',newText,'HorizontalAlignment','left','FontSize',12);
  nRows = numel(newText);
  if (nRows < 2),
    sliderEnable = 'off';
  else
    sliderEnable = 'on';
  end
  nRows = max(nRows-1,1);
  set(hSlider,'Enable',sliderEnable,'Max',nRows,...
      'SliderStep',[1 3]./nRows,'Value',nRows);  